
/* Erstellt: 2022-05-10
 * Version V1.2
 * Autor:  Roland Kaiser, OE2ROL
 * Geschrieben für Arduino Nano ATmega 328P(Old Bootloader)
 * Der Autor übernimmt keine Haftung etwaiger Folgeschäden an Transceivern oder dgl.
 * 
 * Vielen Dank auch an Balazs Kelemen der die Library LcdBarGraph.h erstellt hat.
 *  
 * Diese Software ist für den eigenen privaten Gebrauch frei verfügbar, jedoch nicht für kommerzielle Zwecke ohne Genehmigung des Autors.
 * Veränderungen und Verbesserungen der Software bitte wieder an den Autor Roland Kaiser OE2ROL (roland.kaiser@sbg.at) senden.
 * Die Software wurde am ICOM IC-9700 getestet.
 * 
 * Funktionsweise:
 *   
 * Die 2m Frequenz zum Upconverter wird an der CAT-Schnittstelle des Transceivers gelesen
 * Die 70cm Frequenz vom Downconverter wird an der CAT-Schnittstelle des Transceivers gelesen
 * Berechnung der SAT-Sendefrequenz (Ausgangsfrequenz + TX-Oszillatorfrequenz) und Ausgabe auf ein LCD Display 
 * Berechnung der SAT-Empfangsfrequenz (Eingangsfrequenz + RX-Oszillatorfrequenz) und Ausgabe auf ein LCD Display
 * Ausgabe der vor- und rücklaufenden Leistung mit Bargraphanzeige auf dem LCD Display 
 * PWM Temperatursteuerung des Lüfters zur Kühlung der PA 
 * Automatische Einstellung der Sendeleistung im SAT-Mode
 * Steuerung eines Koaxrelais (12V Schaltausgang) zum automatischen Umschalten der Antennenbuchsen zwischen terrestrischen Antennen und im SAT-Mode auf Up- bzw Downconverter 
 * Menüsteuerung zum Setzen der Oszillatorfrequenzen und der dazu entsprechenden Schrittweite
 * Speichern der Oszillatorfrequenzen in das Arduino-interne EEPROM
 *  
 * Änderungsübersicht:
 * V1.0 initiale Version
 * V1.1 Statusanzeige der CI-V Kommunikation mittels "Pulsanzeige" im Display unten rechts ausgeben
 * V1.2 Ausgabe der Version in der Initmeldung
 *      Unterdrückung einer falschen Frequenzanzeige bei manueller Auswahl der Mainfrequenz
 *      Oszillatorfrequenzen vor Berechnung aus dem EEPROM lesen
 *      
 * Anschlussbelegung der Ports:
   
   CAT port GND -> Arduino GND
   CAT port TX,RX  -> Arduino pin 11,12
   
   Tastenport up      -> Arduino Pin A3
   Tastenport down    -> Arduino Pin A4
   Tastenport Menü    -> Arduino Pin A5
   Die Tasten werden gegen Masse angeschlossen.

   Display:     Arduino:
   4(RS)    ->  7
   6(E)     ->  10
   11(D4)   ->  5
   12(D5)   ->  4
   13(D6)   ->  6
   14(D7)   ->  2
 */

#include <LiquidCrystal.h>
#include <LcdBarGraph.h>
#include <SoftwareSerial.h>
#include <AltSoftSerial.h>
#include <EEPROM.h>

uint32_t RXfreq;                      // umgerechnete Empfangsfrequenz
uint32_t TXfreq;                      // umgerechnete Sendefrequenz
uint32_t TXosz;                       // TX Oszillatorfrequenz
uint32_t RXosz;                       // RX Oszillatorfrequenz
byte civadr = 0xA2;                   // CI-V Adresse
uint32_t subfreq = 0;                 // TX Frequenz am Transceiver
uint32_t mainfreq = 0;                // RX Frequenz am Transceiver
const uint32_t decMulti[]    = {1000000000, 100000000, 10000000, 1000000, 100000, 10000, 1000, 100, 10, 1}; // Multiplikatorarray für Frequenzberechnung
byte lcdNumCols = 20;                 // LCD Anzahl der Stellen
byte FWD = 0;                         // Analogeingang 0 FWD
byte REF = 1;                         // Analogeingang 1 REF
int Temp = 2;                         // Analogeingang 2 Temp
int pwmPinBUZ71 = 3;                  // PWM Ausgang Lüftersteuerung
int temp;                             // Variable für Temperatur
int tempPin = 2;                      // Temperatursensoreingang
const int AntSWPin = 13;              // Pin für Ant SW 
uint32_t freqmem;                     // Frequenzzwischenspeicher
unsigned long previousMillis = 0;     // Timer auf 0 setzen
const long interval = 3500;           // Verzögerungszeit [ms] zur RXfrequenz Berechnung
const int menuPin = A5;               // Pin für Menue Taste
const int downpin = A4;               // Pin für Up Taste
const int uppin = A3;                 // Pin für Down Taste
int menuPushCounter = 0;              // Menütastenzähler
int menuState = 0;                    // Menüstatus
int lastmenuState = 0;                // letzter Menü Status
boolean menubool = false;             // Menü initial deaktivieren
int up;                               // up Taste
int down;                             // Taste down
int menucounter = 0;                  // Menü Zähler
uint32_t sw = 1;                      // Schrittweite zum Einstellen der Oszillatorfrequenzen im Menü 

LiquidCrystal lcd(7, 10, 5, 4, 6, 2); // LCD Anschlussbelegung
LcdBarGraph fwd(&lcd, 16, 22, 2);     // erstellt 16 char bargraph 4.Stelle, 3.Zeile (Stelle 2)
LcdBarGraph ref(&lcd, 16, 22, 3);     // erstellt 16 char bargraph 4.Stelle, 4.Zeile (Stelle 3)
SoftwareSerial CAT(11, 12);           // RX, TX Pin CAT Schnittstelle
AltSoftSerial COM1(8, 9);             // RX, TX TTL (USB) Schnittstelle



void setup(){  
CAT.begin(9600);                      // CAT schnittstelle initialisieren
Serial.begin(9600);                   // Serial Monitor Init
lcd.begin(2, lcdNumCols);             // LCD init
lcd.clear();                          // LCD löschen
delay(100);                           // Verzögerung
pinMode(AntSWPin, OUTPUT);            // Setzt den Analogpin A6 als Outputpin
pinMode(menuPin, INPUT);              // Init für die Tasten Pins
digitalWrite(menuPin, HIGH);          // Init für die Tasten Pins
pinMode(downpin, INPUT);              // Init für die Tasten Pins
digitalWrite(downpin, HIGH);          // Init für die Tasten Pins
pinMode(uppin, INPUT);                // Init für die Tasten Pins
digitalWrite(uppin, HIGH);            // Init für die Tasten Pins

// Initmeldung
lcd.clear();                          // LCD löschen
lcd.setCursor(0, 0);                  // 1. Zeichen, 1. Zeile
lcd.print("QO-100 SAT Converter");    // schreiben
lcd.setCursor(0, 1);                  // 1. Zeichen, 2. Zeile
lcd.print("  **** OE2ROL ****  ");    // schreiben
lcd.setCursor(18, 3);                 // 1. Zeichen, 4. Zeile
lcd.print("V1.2");                    // schreiben
delay(2000);                          // Wartezeit
lcd.clear();                          // LCD löschen

/*
// ************   Wird nur bei der  1. Initialisierung einen neuen Arduino verwendet dann deaktivieren um die Offsets aus dem eeprom zu verwenden **************
TXosz = 2256000000;                   // [Hz] = 2256 MHz
RXosz = 1005400000;                   // [Hz/10] = 10,054 GHz
EEPROM.put(0, TXosz);                 // Wert TXosz ins eeprom schreiben (Adresse 0) 
delay(200);                           // Verzögerung 200 ms
EEPROM.put(10, RXosz);                // Wert RXosz ins eeprom schreiben (Adresse 10)
delay(200);                           // Verzögerung 200 ms
*/
}


// Analogwert vom Temperatursensor einlesen und entsprechende PWM für Lüftersteuerung generieren

void fan() {
  temp = analogRead(tempPin);
  int mv = ( temp/1024.0)*5000;
  int cel = mv/10;
  TCCR2B = TCCR2B & B11111000 | 0x01;   // Timer 2 Teiler auf 1 setzen für PWM Frequenz 31300 Hz
  analogWrite(pwmPinBUZ71, mv/35);      // hier kann die Ansprechschwelle der Lüftersteuerung verändert werden (mv/xx)


  Serial.print("TEMP. = ");
  Serial.print(cel);
  Serial.print("°C");
  Serial.print(" ");
  Serial.print(mv/25);
  Serial.println();

 
delay(100);                           // Verzögerung
}



// Statusanzeige der CI-V Kommunikation

void puls(){
  lcd.setCursor(37, 3);               // Cursor setzen
  lcd.print("+");                     // Zeichen ausgeben
  delay(100);                         // Verzögerung
  lcd.setCursor(37, 3);               // Cursor setzen
  lcd.print(" ");                     // Zeichen ausgeben
  delay(100);                         // Verzögerung
  lcd.setCursor(37, 3);               // Cursor setzen
  lcd.print("*");                     // Zeichen ausgeben
  delay(100);                         // Verzögerung
}



// Analogwert in Bargraph einlesen

void Bargraph(void)
{
  fwd.drawValue( analogRead(FWD), 256); // schreibe Bargraph aus dem gelesenen Analogwert (max. Auflösung 1024)
  ref.drawValue( analogRead(REF), 256); // schreibe Bargraph aus dem gelesenen Analogwert (max. Auflösung 1024)
  delay(100);                           // Verzögerung
}


// LCD Beschriftung setzen

void LCDBeschriftung(void) {
lcd.setCursor(0, 0);                    // 1. Zeichen, 1. Zeile
lcd.print("RX");                        // schreiben
lcd.setCursor(17, 0);                   // 18. Zeichen, 1. Zeile
lcd.print("MHz");                       // schreiben
lcd.setCursor(0, 1);                    // 1. Zeichen, 2. Zeile
lcd.print("TX");                        // schreiben
lcd.setCursor(17, 1);                   // 18. Zeichen, 2. Zeile
lcd.print("MHz");                       // schreiben
lcd.setCursor(18, 2);                   // 1. Zeichen, 3. Zeile
lcd.print("FWD");                       // schreiben
lcd.setCursor(18, 3);                   // 1. Zeichen, 4. Zeile
lcd.print("REF");                       // schreiben
}



// Sendeleistung am Transceiver setzen

void set_TXPower() {
uint8_t power[] = {0xFE, 0xFE, civadr, 0x00, 0x14, 0x0A, 0x00, 0x00, 0xFD};   // CAT Kommando zum setzen der TX Power (7.Byte 0x00 0x00 = Minimum bis 8.Byte 0x02 0x55 = Maximum)
for (uint8_t i = 0; i < sizeof(power); i++) {                                 // Zählschleife 0 - Grösse des Schreibarrays
  CAT.write(power[i]);                                                        // CAT Kommando senden
  }
delay(20);                                                                    // Verzögerung 20 ms  
}



// MAIN Frequenz am Transceiver auswählen

void select_main() {
uint8_t main[] = {0xFE, 0xFE, civadr, 0x00, 0x07, 0xD0, 0xFD};          // CAT Kommando zum Umschalten auf die Main Frequenz
for (uint8_t i = 0; i < sizeof(main); i++) {                            // Zählschleife 0 - Grösse des Schreibarrays
  CAT.write(main[i]);                                                   // CAT Kommando senden
  }
delay(20);                                                              // Verzögerung 20 ms
}



// SUB Frequenz am Transceiver auswählen

void select_sub() {
uint8_t sub[] = {0xFE, 0xFE, civadr, 0x00, 0x07, 0xD1, 0xFD};           // CAT Kommando zum Umschalten auf die Sub Frequenz
for (uint8_t i = 0; i < sizeof(sub); i++) {                             // Zählschleife 0 - Grösse des Schreibarray
  CAT.write(sub[i]);                                                    // CAT Kommando senden
  }
delay(20);                                                              // Verzögerung 20 ms
}



// SAT MODE vom Transceiver auslesen und entsprechende Funktionen ausführen

void read_satmode() {
uint8_t satmode_buffer[12];                                             // Array zum Lesen des Sat-Modes
uint8_t satmode[] = {0xFE, 0xFE, civadr, 0x00, 0x16, 0x5A, 0xFD};       // CAT Kommando zum auslesen der Frequenz  
for (uint8_t i = 0; i < sizeof(satmode); i++) {                         // Grösse des Schreibarrays 
  CAT.write(satmode[i]);                                                // CAT Kommando senden
  }
delay(20);                                                              // Verzögerung 20 ms
  while (!CAT.available());                                             // auf Serial Port warten
    while (CAT.available() > 0) {                                       // wenn Serial Port verfügbar
      for (int j = 0; j < 12; j++) {                                    // Zählschleife 0-12
        satmode_buffer[j] = CAT.read();                                 // Zeichen vom Serial Port schrittweise in das Array schreiben
          if (satmode_buffer[j] == 0xFE){                               // wenn 0xFE im Buffer steht
            j = 0;                                                      // Arrayindexzähler auf 0 setzen
          }
      }                                                                 // FE FE 00 A2 <01> FD = Sat Mode ein, <00> Sat Mode aus
    }
    if (satmode_buffer[5] == 0x01) {                                    // wenn Sat-Mode ein ->
     set_TXPower();                                                     // Setzen der Sendeleistung ausführen
     digitalWrite(AntSWPin, HIGH);                                      // Relaisausgang für Antennenschalter auf High setzen
     LCDBeschriftung();                                                 // LCD Beschriftung ausführen
     print_TXfreq();                                                    // TX Frequenz schreiben
     RX_Freq_berechnen();                                               // RX Frequenz berechnen ausführen
     print_RXfreq();                                                    // RX Frequenz schreiben
     Bargraph();                                                        // Bargraph ausführen
     puls();                                                            // Statusanzeige der CI-V Kommunikation ausgeben 
    }
    else if (satmode_buffer[5] == 0x00) {                               // wenn Sat-Mode aus ->
     digitalWrite(AntSWPin, LOW);                                       // Relaisausgang für Antennenschalter auf Low setzen
     lcd.clear();                                                       // LCD löschen
     lcd.setCursor(1, 1);                                               // Cursor setzen
     lcd.print("   no SAT-MODE     ");                                  // Anzeige nicht im SAT-Mode
     puls();                                                            // Statusanzeige der CI-V Kommunikation ausgeben
     } 
} 



// verzögert RX Frequenz berechnen, wenn TX Frequenz am Transceiver geändert wird

void RX_Freq_berechnen() {
int dly = 10;                                                           // Verzögerung für x millisekunden zwischen den Kommandos
unsigned long currentMillis = millis();                                 // Wert für aktuellen Timer setzen 
if (currentMillis - previousMillis >= interval)                         // wenn Differenz des aktuellen Timers zu Startwert des Timers > Verzögerungszeit ist, dann:
  {
   previousMillis = currentMillis;                                      // Timerstartwert auf Wert des aktuellen Timers setzen  
        if (freqmem!=subfreq)                                           // Nur wenn der Zwischenspeicher nicht der Sub-Frequenz entspricht, dann:
          {
           read_TXfreq();                                               // TX Frequenz lesen ausführen
           delay(dly);                                                  // Verzögerung
           freqmem=subfreq;                                             // TX Frequenz in den Zwischenspeicher schreiben
           delay(dly);                                                  // Verzögerung
           read_RXfreq();                                               // RX Frequenz lesen ausführen
          } 
   }
}



// Berechnete TX Frequenz ins Display schreiben

void print_TXfreq() {
TXosz = EEPROM.get(0, TXosz);        // TX Oszillator [Hz] aus dem eeprom lesen (Adresse 0)  
TXfreq=(subfreq)+(TXosz);            // Anzuzeigende TX Frequenz berechnen
if (TXfreq < 2401000000)             // Zur Unterdrückung einer falschen Frequenzanzeige bei manueller Auswahl der Mainfrequenz 
    {
    lcd.setCursor(4, 1);             // Cursor setzen
    lcd.print(TXfreq/1000000000%10); // Tx 1G Stelle ausgeben
    lcd.setCursor(5, 1);             // Cursor setzen
    lcd.print(TXfreq/100000000%10);  // Tx 100M  Stelle ausgeben
    lcd.setCursor(6, 1);             // Cursor setzen
    lcd.print(TXfreq/10000000%10);   // Tx 10M Stelle ausgeben
    lcd.setCursor(7, 1);             // Cursor setzen
    lcd.print(TXfreq/1000000%10);    // Tx 1M Stelle ausgeben
    lcd.setCursor(8, 1);             // Cursor setzen
    lcd.print(",");                  // , ausgeben
    lcd.setCursor(9, 1);             // Cursor setzen
    lcd.print(TXfreq/100000%10);     // Tx 100k Stelle ausgeben
    lcd.setCursor(10, 1);            // Cursor setzen
    lcd.print(TXfreq/10000%10);      // Tx 10k Stelle ausgeben
    lcd.setCursor(11, 1);            // Cursor setzen
    lcd.print(TXfreq/1000%10);       // Tx 1k Stelle ausgeben
    lcd.setCursor(12, 1);            // Cursor setzen
    lcd.print(TXfreq/100%10);        // Tx 100Hz Stelle ausgeben
    lcd.setCursor(13, 1);            // Cursor setzen
    lcd.print(TXfreq/10%10);         // Tx 10Hz Stelle ausgeben
    lcd.setCursor(14, 1);            // Cursor setzen
    lcd.print(TXfreq%10);            // Tx 1Hz Stelle ausgeben
    }
}



// Berechnete RX Frequenz ins Display schreiben

void print_RXfreq() {
RXosz = EEPROM.get(10, RXosz);   // RX Oszillator [Hz] aus dem eeprom lesen (Adresse 10)
RXfreq=((mainfreq/10))+(RXosz);  // Anzuzeigende RX Frequenz berechnen
lcd.setCursor(1, 2);             // Cursor setzen
lcd.print(RXfreq/1000000000%10); // Rx 10G Stelle ausgeben
lcd.setCursor(2, 2);             // Cursor setzen
lcd.print(RXfreq/100000000%10);  // Rx 1G Stelle ausgeben
lcd.setCursor(3, 2);             // Cursor setzen
lcd.print(RXfreq/10000000%10);   // Rx 100M Stelle ausgeben
lcd.setCursor(4, 2);             // Cursor setzen
lcd.print(RXfreq/1000000%10);    // Rx 10M Stelle ausgeben
lcd.setCursor(5, 2);             // Cursor setzen
lcd.print(RXfreq/100000%10);     // Rx 1M Stelle ausgeben
lcd.setCursor(6, 2);             // Cursor setzen
lcd.print(",");                  // , ausgeben
lcd.setCursor(7, 2);             // Cursor setzen
lcd.print(RXfreq/10000%10);      // Rx 100k Stelle ausgeben
lcd.setCursor(8, 2);             // Cursor setzen
lcd.print(RXfreq/1000%10);       // Rx 10k Stelle ausgeben
lcd.setCursor(9, 2);             // Cursor setzen
lcd.print(RXfreq/100%10);        // Rx 1k Stelle ausgeben
lcd.setCursor(10, 2);            // Cursor setzen
lcd.print(RXfreq/10%10);         // Tx 100Hz Stelle ausgeben
lcd.setCursor(11, 2);            // Cursor setzen
lcd.print(RXfreq%10);            // Tx 10Hz Stelle ausgeben
lcd.setCursor(12, 2);            // Cursor setzen
lcd.print(TXfreq%10);            // Tx 1Hz Stelle ausgeben 
}



// RX Frequenz aus dem Transceiver lesen (MAIN)

void read_RXfreq() {                                                      // Frequenz vom Transceiver auslesen und in Variable (freq) schreiben
select_main();                                                            // Main Frequenz auswählen
uint8_t freq_buffer[12];                                                  // Array zum Einlesen der Main Frequenz
uint8_t req[] = {0xFE, 0xFE, civadr, 0x00, 0x03, 0xFD};                   // CAT Kommando zum auslesen der Frequenz  
for (uint8_t i = 0; i < sizeof(req); i++) {                               // Grösse des Schreibarrays 
  CAT.write(req[i]);
  }
delay(20);                                                                // Verzögerung 20 ms
  while (!CAT.available());                                               // auf Serial Port warten
    while (CAT.available() > 0) {                                         // wenn Serial Port verfügbar
      for (int j = 0; j < 12; j++) {                                      // Zählschleife 0-12
        freq_buffer[j] = CAT.read();                                      // Zeichen vom Serial Port schrittweise in das Array schreiben
            if (freq_buffer[j] == 0xFF){                                  // wenn 0xFF im Buffer steht
            j = 0;                                                        // Arrayindexzähler auf 0 setzen
            } 
        mainfreq = 0;                                                     // Variable für TX Frequenz auf 0 setzen
                                                                          // FE FE 00 A2 03 <00 90 42 44 01> FD = 0144429000 Hz
        for (uint8_t k = 0; k < 5; k++) {                                 // Zählschleife 0-5
          mainfreq += (freq_buffer[10 - k] >> 4) * decMulti[k * 2];       // Frequenz aus dem Buffer lesen und in Dezimal konvertieren
          mainfreq += (freq_buffer[10 - k] & 0x0F) * decMulti[k * 2 + 1]; // Frequenz aus dem Buffer lesen und in Dezimal konvertieren
          }
        }
      }
select_sub(); 
} 



// TX Frequenz aus dem Transceiver lesen (SUB)

void read_TXfreq() {                                                      // Frequenz vom Transceiver auslesen und in Variable (freq) schreiben
uint8_t freq_buffer[12];                                                  // Array zum Einlesen der Main Frequenz
uint8_t req[] = {0xFE, 0xFE, civadr, 0x00, 0x03, 0xFD};                   // CAT Kommando zum auslesen der Frequenz  
for (uint8_t i = 0; i < sizeof(req); i++) {                               // Grösse des Schreibarrays 
  CAT.write(req[i]);
  }
delay(20);                                                                // Verzögerung 20 ms
  while (!CAT.available());                                               // auf Serial Port warten
    while (CAT.available() > 0) {                                         // wenn Serial Port verfügbar
      for (int j = 0; j < 12; j++) {                                      // Zählschleife 0-12
        freq_buffer[j] = CAT.read();                                      // Zeichen vom Serial Port schrittweise in das Array schreiben
            if (freq_buffer[j] == 0xFF){                                  // wenn 0xFF im Buffer steht
            j = 0;                                                        // Arrayindexzähler auf 0 setzen
            }
        subfreq = 0;                                                      // Variable für TX Frequenz auf 0 setzen
                                                                          // FE FE 00 A2 03 <00 90 42 44 01> FD = 0144429000 Hz
        for (uint8_t k = 0; k < 5; k++) {                                 // Zählschleife 0-5
          subfreq += (freq_buffer[10 - k] >> 4) * decMulti[k * 2];        // Frequenz aus dem Buffer lesen und in Dezimal konvertieren
          subfreq += (freq_buffer[10 - k] & 0x0F) * decMulti[k * 2 + 1];  // Frequenz aus dem Buffer lesen und in Dezimal konvertieren
          }
        }
      }
} 



// Softwarereset

void software_reset(){
up = digitalRead(uppin);                                // up Taste abfragen
down = digitalRead(downpin);                            // down Taste abfragen
            if (up == LOW && down == LOW){              // wenn up und down Taste gleichzeitig betätigt
              asm volatile ("  jmp 0");                 // Software Reset
            }
}



// Menüsteuerung

void menuset(){
menuState = digitalRead(menuPin);                       // Auslesen der Menütaste
  if (menuState != lastmenuState)                       // wenn sich der Menü Status geändert hat
  {
    if (menuState == LOW)                               // wenn Menü Status LOW (Menütaste gedrückt)
    {
      delay(200);                                       // Verzögerung
      menuPushCounter = 4;                              // maximal 4 Menüs verfügbar Menütastenzähler auf 4 setzen
      menubool = true;                                  // menü aktivieren
    }
    if (menuState == HIGH)                              // wenn Menü Status HIGH
    {
      menubool = false;                                 // menü deaktivieren
    }
  }

  if (menubool == true && menuPushCounter == 4)         // wenn Menü aktiviert und Menütastenzähler 4
  {
    menucounter = 0;                                    // Menüzähler auf 0 setzen 
    menu();                                             // Menü starten
  }
lastmenuState = menuState;                              // Menüstatus setzen
}



// Menüsteuerung zum Setzen und Speichern der Offsets

int menu()
{
  lcd.clear();                                    // LCD löschen
  delay(100);                                     // Verzögerung
  while (menucounter < 4)                         // while Menu enspricht der max Anzahl der Menüs
  {
    menuState = digitalRead(menuPin);             // Menü Taste abfragen
    up = digitalRead(uppin);                      // up Taste abfragen
    down = digitalRead(downpin);                  // down Taste abfragen

    if (menuState != lastmenuState)               // wenn sich der Menü Status geändert hat
    {
      lcd.clear();                                // LCD löschen
      if (menuState == LOW)                       // wenn Menü Status LOW
      {
        delay(200);                               // Verzögerung
        menucounter++;                            // Menüzähler um 1 erhöhen
      }
    }
lastmenuState = menuState;                        // letzten Menüstatus auf Menü Status setzen

    switch (menucounter) {                        // Menüauswahl
      case 0:                                     // Menü 1 TX Offset
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("TX Oscillator [Hz]");          // schreiben
        lcd.setCursor(0, 1);                      // 1. Zeichen, 2. Zeile
        lcd.print(TXosz);                         // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              TXosz=TXosz+(sw*10);                // Wert um Schrittweite *10 erhöhen
              lcd.clear();                        // LCD löschen
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              delay(200);                         // Verzögerung
              TXosz=TXosz-(sw*10);                // Wert um Schrittweite *10 vermindern
              lcd.clear();                        // LCD löschen
            }
        break;                                    // aussteigen, dieses Menü beginnt dann von vorne

      case 1:                                     // Menü 2 RX Offset
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("RX Oscillator[Hz/10]");        // schreiben
        lcd.setCursor(0, 1);                      // 1. Zeichen, 2. Zeile
        lcd.print(RXosz);                         // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              RXosz=RXosz+sw;                     // Wert um Schrittweite erhöhen
              lcd.clear();                        // LCD löschen
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              delay(200);                         // Verzögerung
              RXosz=RXosz-sw;                     // Wert um Schrittweite vermindern
              lcd.clear();                        // LCD löschen
            }
        break;                                    // aussteigen, dieses Menü beginnt dann von vorne
        
      case 2:                                     // Menü 2 Split Offset
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("Step size");                   // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              sw=sw*10;                           // Wert mit 10 multiplizieren
              lcd.clear();                        // LCD löschen
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              delay(200);                         // Verzögerung
              sw=sw/10;                           // Wert durch 10 dividieren
              lcd.clear();                        // LCD löschen
            }
            if (sw*10 > 100000000) {sw=1;}        // max Begrenzung auf 100M
            if (sw < 1) {sw=1;}                   // min Begrenzung auf 1
            if (sw*10 < 1000){                    // Beschriftung und Umrechnung in Hz
              lcd.setCursor(0, 1);                // 1. Zeichen, 2. Zeile
              lcd.print(sw*10);                   // schreiben   
              lcd.setCursor(5, 1);                // 5. Zeichen, 2. Zeile
              lcd.print("Hz");                    // schreiben
            }
            if (sw*10 > 999 && sw*10 < 1000000){  // Beschriftung und Umrechnung in KHz
              lcd.setCursor(0, 1);                // 1. Zeichen, 2. Zeile
              lcd.print(sw*10/1000);              // schreiben
              lcd.setCursor(5, 1);                // 5. Zeichen, 2. Zeile
              lcd.print("KHz");                   // schreiben
            }
            if (sw*10 > 999999 && sw*10 < 1000000000){ // Beschriftung und Umrechnung in MHz
              lcd.setCursor(0, 1);                // 1. Zeichen, 2. Zeile
              lcd.print(sw*10/1000000);           // schreiben
              lcd.setCursor(5, 1);                // 5. Zeichen, 2. Zeile
              lcd.print("MHz");                   // schreiben
            }
        break;                                    // aussteigen, dieses Menü beginnt dann von vorne
        
      case 3:                                     // Menü 3, hier werden die Werte gespeichert und der arduino beginnt loop neu auszuführen
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("store Offsets?");              // schreiben
        lcd.setCursor(0, 1);                      // 1. Zeichen, 2. Zeile
        lcd.print("down or menu= No");            // schreiben
        lcd.setCursor(18, 2);                     // 1. Zeichen, 3. Zeile
        lcd.print("up= Yes");                     // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              EEPROM.put(0, TXosz);               // Wert ins eeprom schreiben (Adresse 0)
              delay(200);                         // Verzögerung
              EEPROM.put(10, RXosz);              // Wert ins eeprom schreiben (Adresse 10)
              delay(200);                         // Verzögerung
              lcd.setCursor(0, 1);                // 1. Zeichen, 1. Zeile
              delay(200);                         // Verzögerung
              lcd.clear();                        // LCD löschen
              lcd.print("Offsets stored");        // schreiben
              delay(2000);                        // Verzögerung
              lcd.clear();                        // LCD löschen
              delay(200);                         // Verzögerung
              menucounter = 4;                    // Menü Zähler auf 4 setzen
              menuPushCounter = 0;                // Menütastenzähler auf 0 setzen
              break;                              // aussteigen, zurück zu loop
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              lcd.clear();                        // LCD löschen
              delay(200);                         // Verzögerung
              menucounter = 4;                    // Menü Zähler auf 4 setzen
              menuPushCounter = 0;                // Menütastenzähler auf 0 setzen
              break;                              // aussteigen, zurück zu loop
            }
    }
  }
}



void loop(){
menuset();                                              // Menü ausführen
read_satmode();                                         // Sat Mode lesen ausführen
read_TXfreq();                                          // TX Frequenz lesen ausführen
fan();                                                  // Lüftersteuerung ausführen
software_reset();                                       // Software Reset ausführen
delay(100);                                             // Verzögerung
}
