/*
   Erstellt:  2023-12-10
   Version V2.1
   Autor:  Roland Kaiser, OE2ROL
   geschrieben für Arduino Nano ATmega 328P(Old Bootloader)
   Der Autor übernimmt keine Haftung etwaiger Folgeschäden an Transceivern oder dgl.
   
   Diese Software ist für den eigenen privaten Gebrauch frei verfügbar, jedoch nicht für kommerzielle Zwecke ohne Genehmigung des Autors.
   Veränderungen und Verbesserungen der Software bitte wieder an den Autor Roland Kaiser OE2ROL (roland.kaiser@sbg.at) senden.
   
   Die Software wurde am Yaesu FT-991A getestet, sollte auch am FT-991 funktionieren.

   Funktionsweise:
   
   o  Der Transceiver wird beim Start automatisch in den entsptechenden "SAT Mode" gebracht (beide VFO´s in die vorgegebene QRG, Betriebsart USB, Splitmode, HF-Power 5W)
   o  Der RX VFO wird gelesen, davon wird ein vorgegebener Versatz subtrahiert und verzögert automatisch in den TX VFO geschrieben.
   o  Dies passiert vollautomatisch nur durch verändern des RX VFO´s.
   o  Die für QO-100 umgerechnete RX und TX Frequenz wird auf einem LCD Display angezeigt. 
   o  RIT kann mit Tasten up oder down um voreingestellte Schrittweite verstellt werden (hier wird nur die Empfangsfrequenz verändert)
   o  Menüsteuerung mit 5 Ebenen:
       o Setzen der RX-Oszillatorfrequenz mit voreingestellter Schrittweite zum Anpassen der Oszillatorfrequenz des Upconverters
       o Setzen der TX-Oszillatorfrequenz mit voreingestellter Schrittweite zum Anpassen der Oszillatorfrequenz des Downconverters
       o Setzen eines Split-Offsets mit voreingestellter Schrittweite für Stationen die im Split-Betrieb arbeiten (hier wird nur die Sendefrequenz verändert)
       o Setzen der Schrittweite für alle Einstellungen
       o Speichern der RX/TX Oszillatorfrequenzen im internen EEPROM.
   o  Nach dem Speichern der Offsets wird bei einem Neustart der SAT BOX auf die Frequenz der mittleren QO-100 Bake gestellt.
   o  Die RX und TX Oszillatorfrequenzen bleiben auch ohne Stromversorgung gespeichert, der Split Offset wird wieder auf 0 gesetzt.
   o  Nach erneuten Start werden die Offsets automatisch aus dem EEPROM ausgelesen und für die Korrektur des Up- und Down-Konverters verwendet.
   o  Softwarereset kann mit Up- und Down-Taste gleichzeitig betätigen ausgelöst werden. (nur in der Frequenzanzeige)

   Änderungsübersicht:

   V2.1 Automatisches setzen der Oszillatorfrequenzen bei der  1. Initialisierung eines neuen Arduino
   V2.0 TX RX frei wählbar mit Eingabe entsprechender Oszillatorfrequenzen, Anzeige RX,TX Status hinzugefügt, wählbare Schrittweite 
   V1.0 Initiale Version

   Anschlussbelegung der Ports:
   
   RS232 Radio CAT port GND -> Arduino GND
   RS232 Radio CAT port TX  -> Arduino pin 11
   RS232 Radio CAT port RX  -> Arduino pin 12

   Tastenport up      -> Arduino Pin A3
   Tastenport down    -> Arduino Pin A4
   Tastenport Menü    -> Arduino Pin A5
   Die Tasten werden gegen Masse angeschlossen.

   Display:     Arduino:
   4(RS)    ->  7
   6(E)     ->  10
   11(D4)   ->  5
   12(D5)   ->  4
   13(D6)   ->  6
   14(D7)   ->  2
*/

#include <LiquidCrystal.h>
#include <EEPROM.h>
#include <SoftwareSerial.h>

SoftwareSerial RS232(11, 12);         // RS232 TX Pin 11, RX Pin 12
LiquidCrystal lcd(7, 10, 5, 4, 6, 2); // LCD Anschlussbelegung
 
uint32_t cat_rec_vfo_a = 0;       // empfangene Frequenz von VFO A
uint32_t cat_rec_vfo_b = 0;       // empfangene Frequenz von VFO B
byte lcdNumCols = 20;             // LCD Anzahl der Stellen
uint32_t VFOB;                    // zu TX VFO Frequenz  mit  tx_vfo_b(); an Transceiver senden
uint32_t VFOA;                    // zu RX VFO Frequenz  mit  tx_vfo_a(); an Transceiver senden
uint32_t TXfreq;                  // umgerechnete Sendefrequenz zur Anzeige am LCD Display
uint32_t RXfreq;                  // umgerechnete Empfangsfrequenz zur Anzeige am LCD Display
uint32_t TXosz;                   // TX Oszillatorfrequenz
uint32_t RXosz;                   // RX Oszillatorfrequenz
uint32_t shift;                   // RX-TX Frequenzversatz
uint32_t freqmem;                 // Frequenzzwischenspeicher
long RIT = 0;                     // RIT
long Splitoffset;                 // Split Offset
const int menuPin = A5;           // Pin für Menue Taste
const int downpin = A4;           // Pin für Up Taste
const int uppin = A3;             // Pin für Down Taste
int menuPushCounter = 0;          // Menütastenzähler
int menuState = 0;                // Menüstatus
int lastmenuState = 0;            // letzter Menü Status
boolean menubool = false;         // Menü initial deaktivieren
int up;                           // up Taste
int down;                         // Taste down
int menucounter = 0;              // Menü Zähler
unsigned long previousMillis = 0; // Timer auf 0 setzen
const long interval = 1;          // Verzögerungszeit [ms]zum VFO nachziehen
uint32_t sw = 10;                 // Schrittweite zum Einstellen der Frequenzen im Menü Hz/10 (default 100Hz)
uint32_t state;                   // RX,TX Status 0=RX, 1=TX


// ******************************************SETUP**************************************************************************************************                
void setup() {

  EEPROM.get(0, TXosz);
  if(TXosz == -1){
    // Wird nur bei der  1. Initialisierung eines neuen Arduino verwendet
    TXosz=2256000000;                               // TX Oszillatorfrequenz [Hz]    (z.B. 2256MHz)
    RXosz=1005770000;                               // RX Oszillatorfrequenz [Hz/10] (z.B. 10,054GHz)
    EEPROM.put(0, TXosz);                           // Wert TXosz ins EEPROM schreiben (Adresse 0) 
    delay(200);                                     // Verzögerung
    EEPROM.put(10, RXosz);                          // Wert RXosz ins EEPROM schreiben (Adresse 10)
    delay(200);                                     // Verzögerung
  }
  
Serial.begin(9600);                             // Serial.begin(9600) für Testzwecke 
while (!Serial);                                // wenn Serielle Schnittstelle verfügbar
RS232.begin(9600);                              // Init Software Serial
pinMode(menuPin, INPUT);                        // Init für die Tasten Pins
digitalWrite(menuPin, HIGH);                    // Init für die Tasten Pins
pinMode(downpin, INPUT);                        // Init für die Tasten Pins
digitalWrite(downpin, HIGH);                    // Init für die Tasten Pins
pinMode(uppin, INPUT);                          // Init für die Tasten Pins
digitalWrite(uppin, HIGH);                      // Init für die Tasten Pins
lcd.begin(2, lcdNumCols);                       // LCD init
lcd.clear();                                    // LCD löschen
delay(100);                                     // Verzögerung


// **************************************Initmeldung************************************************************************************************
lcd.setCursor(0, 0);                            // 1. Zeichen, 1. Zeile
lcd.print(" **** SAT  BOX ****");               // schreiben
lcd.setCursor(0, 1);                            // 1. Zeichen, 2. Zeile
lcd.print("  **** OE2ROL ****");                // schreiben
lcd.setCursor(18, 2);                           // 1. Zeichen, 3. Zeile
lcd.print("FT-991A V2.1");                      // schreiben
delay(2000);                                    // Wartezeit
lcd.clear();                                    // LCD löschen


// *****************************************Transceiver Settings setzen*****************************************************************************
int dly = 100;                                  // Verzögerung für x millisekunden zwischen den Kommandos
RXosz = EEPROM.get(10, RXosz);                  // RX Oszillatorfrequenz [Hz/10] aus dem eeprom lesen
TXosz = EEPROM.get(0, TXosz);                   // TX Oszillatorfrequenz [Hz] aus dem eeprom lesen
RS232.print("QS;");                             // Split
delay(dly);                                     // Verzögerung dly [ms]
RS232.print("PC005;");                          // Power 5W
delay(dly);                                     // Verzögerung dly [ms]
RS232.print("BA;");                             // VFO B zu A
delay(dly);                                     // Verzögerung dly [ms]
VFOA=(1048975000-RXosz)*10;                     // VFO A Startfrequenz setzen (mittlere Bake QO-100)
tx_vfo_a();                                     // VFO A schreiben
delay(dly);                                     // Verzögerung dly [ms]
RS232.print("MD02;");                           // Mode USB
delay(dly);                                     // Verzögerung dly [ms]
RS232.print("PC005;");                          // Power 5W
delay(dly);                                     // Verzögerung dly [ms]
RS232.print("AB;");                             // VFO A zu B
delay(dly);                                     // Verzögerung dly [ms]
VFOB=(2400250000-TXosz);                        // VFO B Startfrequenz setzen (mittlere Bake QO-100)
tx_vfo_b();                                     // VFO B schreiben
delay(dly);                                     // Verzögerung dly [ms]
RS232.print("MD02;");                           // Mode USB
delay(dly);                                     // Verzögerung dly [ms]
RS232.print("PC005;");                          // Power 5W
delay(dly);                                     // Verzögerung dly [ms]
}


// ********************************************************Softwarereset****************************************************************************
void software_reset(){
up = digitalRead(uppin);                                // up Taste abfragen
down = digitalRead(downpin);                            // down Taste abfragen
            if (up == LOW && down == LOW){              // wenn up und down Taste gleichzeitig betätigt
              asm volatile ("  jmp 0");                 // Software Reset
            }
}


// ********************************************LCD Beschriftung setzen******************************************************************************
void LCDBeschriftung() {
lcd.setCursor(1, 0);      // 2. Zeichen, 1. Zeile
lcd.print("RX");          // schreiben
lcd.setCursor(17, 0);     // 18. Zeichen, 1. Zeile
lcd.print("MHz");         // schreiben
lcd.setCursor(1, 1);      // 2. Zeichen, 2. Zeile
lcd.print("TX");          // schreiben
lcd.setCursor(17, 1);     // 18. Zeichen, 2. Zeile
lcd.print("MHz");         // schreiben
}


//*************************************************************RX_TX Beschriftung*******************************************************************
void RXTXBeschriftung() {
  if (state == 0) {           // wenn state 0 ist (RX) dann
    lcd.setCursor(0, 0);      // 1. Zeichen, 1. Zeile
    lcd.print("*");           // schreiben
  }
  else{                       // sonst
    lcd.setCursor(0, 0);      // 1. Zeichen, 1. Zeile
    lcd.print(" ");           // schreiben
  }
  if (state == 1) {           // wenn state 1 ist (TX) dann
    lcd.setCursor(0, 1);      // 1. Zeichen, 2. Zeile
    lcd.print("*");           // schreiben
  }
  else{                       // sonst
    lcd.setCursor(0, 1);      // 1. Zeichen, 2. Zeile
    lcd.print(" ");           // schreiben
  }
}


// *********************************************Sende- und Empangsfrequenzen berechnen**************************************************************
void calculateFrequency()  {
TXosz = EEPROM.get(0, TXosz);               // TX Oszillatorfrequenz [Hz] aus dem eeprom lesen (Adresse 0)
RXosz = EEPROM.get(10, RXosz);              // RX Oszillatorfrequenz [Hz*10] aus dem eeprom lesen (Adresse 10)
shift=808950000;                            // QO-100 RX-TX Versatz [Hz/10] = TX 10489,750MHz RX 2400,250MHz = RX-TX shift 8089,500MHz
TXfreq=VFOB+TXosz;                          // Berechnung der SAT Sendefrequenz [Hz](Sendefrequenz + TX Oszillatorfrequenz)
RXfreq=(VFOA/10)+RXosz;                     // Berechnung der SAT Empfangsfrequenz [Hz/10](Empfangsfrequenz/10 + RX Oszillatorfrequenz)
}


// *************************************Berechnete TX Frequenz ins Display schreiben (Uplink)********************************************************
void displayTXfreq() {
lcd.setCursor(5, 1);
lcd.print(TXfreq/1000000000%10);   // Tx 1G Stelle ausgeben
lcd.setCursor(6, 1);
lcd.print(TXfreq/100000000%10);    // Tx 100M  Stelle ausgeben
lcd.setCursor(7, 1);
lcd.print(TXfreq/10000000%10);     // Tx 10M Stelle ausgeben
lcd.setCursor(8, 1);
lcd.print(TXfreq/1000000%10);      // Tx 1M Stelle ausgeben
lcd.setCursor(9, 1);
lcd.print(",");                    // , ausgeben
lcd.setCursor(10, 1);
lcd.print(TXfreq/100000%10);       // Tx 100k Stelle ausgeben
lcd.setCursor(11, 1);
lcd.print(TXfreq/10000%10);        // Tx 10k Stelle ausgeben
lcd.setCursor(12, 1);
lcd.print(TXfreq/1000%10);         // Tx 1k Stelle ausgeben
lcd.setCursor(13, 1);
lcd.print(TXfreq/100%10);          // Tx 100Hz Stelle ausgeben
lcd.setCursor(14, 1);
lcd.print(TXfreq/10%10);           // Tx 10Hz Stelle ausgeben
lcd.setCursor(15, 1);
lcd.print(TXfreq%10);              // Tx 1Hz Stelle ausgeben
}


// *************************************Berechnete RX Frequenz ins Display schreiben (Downlink)****************************************************** 
void displayRXfreq() {
lcd.setCursor(2, 2);
lcd.print(RXfreq/1000000000%10);   // Rx 10G Stelle ausgeben
lcd.setCursor(3, 2);
lcd.print(RXfreq/100000000%10);    // Rx 1G Stelle ausgeben
lcd.setCursor(4, 2);
lcd.print(RXfreq/10000000%10);     // Rx 100M Stelle ausgeben
lcd.setCursor(5, 2);
lcd.print(RXfreq/1000000%10);      // Rx 10M Stelle ausgeben
lcd.setCursor(6, 2);
lcd.print(RXfreq/100000%10);       // Rx 1M Stelle ausgeben
lcd.setCursor(7, 2);
lcd.print(",");                    // , ausgeben
lcd.setCursor(8, 2);
lcd.print(RXfreq/10000%10);        // Rx 100k Stelle ausgeben
lcd.setCursor(9, 2);
lcd.print(RXfreq/1000%10);         // Rx 10k Stelle ausgeben
lcd.setCursor(10, 2);
lcd.print(RXfreq/100%10);          // Rx 1k Stelle ausgeben
lcd.setCursor(11, 2);
lcd.print(RXfreq/10%10);           // Rx 100Hz Stelle ausgeben
lcd.setCursor(12, 2);
lcd.print(RXfreq%10);              // Rx 10Hz Stelle ausgeben
lcd.setCursor(13, 2);
lcd.print(RXfreq%1);               // Rx 1Hz Stelle ausgeben
}


//*************************************************************RX_TX Beschriftung*******************************************************************
void TXstatus(){
long TXstatus_buffer[4];                           // Array zum Einlesen des TX Status
int i = 0;                                         // Array Indexzähler auf 0 setzen

  RS232.print("TX;");                              // CAT Kommando zu lesen des TX Status
  delay(20);                                       // Verzögerung 20 ms
  i = 0;                                           // Array Indexzähler auf 0 setzen
  while (!RS232.available());                      // auf Serial Port warten
    while (RS232.available() > 0) {                // wenn Serial Port verfügbar
      TXstatus_buffer[i] = RS232.read();           // Zeichen vom Serial Port schrittweise in das Lese-Array schreiben, gelesene Daten z.B. (TX0;) für RX,  (TX2;) für TX
      i ++;                                        // Arrayindexzähler hochzählen
      }
      TXstatus_buffer[i] = 0;                      // Arrayindexzähler auf 0 setzen
      if (TXstatus_buffer[2] == 48){               // wenn 3.Zeichen im Lese-Array dezimal 48 ist (Zeichen 0), dann
        state = 0;                                 // state auf 0 setzen
      }
      if (TXstatus_buffer[2] == 50){               // wenn 3.Zeichen im Lese-Array dezimal 50 ist (Zeichen 2), dann
        state = 1;                                 // state auf 1 setzen
      }
}


// ********************************************RX Frequenz aus dem Transceiver lesen (VFOA)**********************************************************
void rx_vfo_a() {                                  // VFO A Frequenz vom Transceiver auslesen und in Variable (cat_rec_vfo_a) schreiben

long vfo_a_in_buffer[13];                          // Array zum Einlesen der Frequenz VFO A
int j = 0;                                         // Array Indexzähler auf 0 setzen

  RS232.print("FA;");                              // CAT Kommando zu lesen der Frequenz von VFO A
  delay(20);                                       // Verzögerung 20 ms
  j = 0;                                           // Array Indexzähler auf 0 setzen
  while (!RS232.available());                      // auf Serial Port warten
    while (RS232.available() > 0) {                // wenn Serial Port verfügbar
      vfo_a_in_buffer[j] = RS232.read();           // Zeichen vom Serial Port schrittweise in das Array schreiben (gelesene Daten z.B. FB432050000;)
      j ++;                                        // Arrayindexzähler hochzählen
      }
      vfo_a_in_buffer[j] = 0;                      // Arrayindexzähler auf 0 setzen

cat_rec_vfo_a=((vfo_a_in_buffer[2]-48)*100000000)+ // 1 Stelle 100 MHZ Digit in ASCII Dezimal - 48 = entsprechende Zahl....
              ((vfo_a_in_buffer[3]-48)*10000000)+  // 2 Stelle 10 MHZ
              ((vfo_a_in_buffer[4]-48)*1000000)+   // 3 Stelle 1 MHZ
              ((vfo_a_in_buffer[5]-48)*100000)+    // 4 Stelle 100 KHZ
              ((vfo_a_in_buffer[6]-48)*10000)+     // 5 Stelle 10 KHZ
              ((vfo_a_in_buffer[7]-48)*1000)+      // 6 Stelle 1 KHZ
              ((vfo_a_in_buffer[8]-48)*100)+       // 7 Stelle 100 HZ
              ((vfo_a_in_buffer[9]-48)*10)+        // 8 Stelle 10 HZ
              (vfo_a_in_buffer[10]-48);            // 9 Stelle 1 HZ
}


// ********************************************TX Frequenz aus dem Transceiver lesen (VFOB)**********************************************************
void rx_vfo_b() {                                  // VFO B Frequenz vom Transceiver auslesen und in Variable (cat_rec_vfo_b) schreiben
  
long vfo_b_in_buffer[13];                          // Array zum Einlesen der Frequenz VFO B
int k = 0;                                         // Array Indexzähler auf 0 setzen

  RS232.print("FB;");                              // CAT Kommando zu lesen der Frequenz von VFO B
  delay(20);                                       // Verzögerung 20 ms
  k = 0;                                           // Array Indexzähler auf 0 setzen
  while (!RS232.available());                      // auf Serial Port warten
    while (RS232.available() > 0) {                // wenn Serial Port verfügbar
      vfo_b_in_buffer[k] = RS232.read();           // Zeichen vom Serial Port schrittweise in das Array schreiben (gelesene Daten z.B. FB144500000;)
      k ++;                                        // Arrayindexzähler hochzählen
      }
      vfo_b_in_buffer[k] = 0;                      // Arrayindexzähler auf 0 setzen
     
cat_rec_vfo_b=((vfo_b_in_buffer[2]-48)*100000000)+ // 1 Stelle 100 MHZ Digit in ASCII Dezimal - 48 = entsprechende Zahl....
              ((vfo_b_in_buffer[3]-48)*10000000)+  // 2 Stelle 10 MHZ
              ((vfo_b_in_buffer[4]-48)*1000000)+   // 3 Stelle 1 MHZ
              ((vfo_b_in_buffer[5]-48)*100000)+    // 4 Stelle 100 KHZ
              ((vfo_b_in_buffer[6]-48)*10000)+     // 5 Stelle 10 KHZ
              ((vfo_b_in_buffer[7]-48)*1000)+      // 6 Stelle 1 KHZ
              ((vfo_b_in_buffer[8]-48)*100)+       // 7 Stelle 100 HZ
              ((vfo_b_in_buffer[9]-48)*10)+        // 8 Stelle 10 HZ
              (vfo_b_in_buffer[10]-48);            // 9 Stelle 1 HZ
}


// ***********************************************RX Frequenz in VFOA schreiben********************************************************************
void tx_vfo_a() {                                  // die Variable (VFOA) in den Transceiver in den VFO A schreiben
  
long vfo_a_buffer[10];                             // Array zum Schreiben der Frequenz in VFO A

for (int i = 0; i < 10; i++) {                     // Zählschleife 0-9
     vfo_a_buffer[i] = VFOA-1;                     // VFO Frequenz in das Array schreiben
     vfo_a_buffer[i]++;                            // Arrayindexzähler hochzählen
     delay(10);                                    // Verzögerung 10 ms
     RS232.print("FA");                            // CAT Kommandoteil VFO A schreiben
     if (VFOA < 100000000){                        // wenn VFOA < 100MHz ist, dann
      RS232.print("0");                            // vorangestellte 0 vor der Frequenz einfügen
     }
     RS232.print(vfo_a_buffer[i]);                 // CAT Kommandoteil Frequenz schreiben
     RS232.print(";");                             // CAT Kommandoteil Stopbit
     vfo_a_buffer[i] = 0;                          // Arrayindexzähler auf 0 setzen
     }
}


// ***********************************************TX Frequenz in VFOB schreiben********************************************************************
void tx_vfo_b() {                                  // die Variable (VFOB) in den Transceiver in den VFO B schreiben
  
long vfo_b_buffer[10];                             // Array zum Schreiben der Frequenz in VFO A

for (int i = 0; i < 10; i++) {                     // Zählschleife 0-9
     vfo_b_buffer[i] = VFOB-1;                     // VFO Frequenz in das Array schreiben
     vfo_b_buffer[i]++;                            // Arrayindexzähler hochzählen
     delay(10);                                    // Verzögerung 10 ms
     RS232.print("FB");                            // CAT Kommandoteil VFOB schreiben
     if (VFOB < 100000000){                        // wenn VFOB < 100MHz ist, dann
      RS232.print("0");                            // vorangestellte 0 vor der Frequenz einfügen
     }
     RS232.print(vfo_b_buffer[i]);                 // CAT Kommandoteil Frequenz schreiben
     RS232.print(";");                             // CAT Kommandoteil Stopbit
     vfo_b_buffer[i] = 0;                          // Arrayindexzähler auf 0 setzen
     }
}


// ************************************VFOB nachziehen, nur wenn RX Frequenz am Transceiver geändert wird*******************************************
void VFOnachziehen() {
int dly = 10;                                                         // Verzögerung für x millisekunden zwischen den Kommandos
rx_vfo_a();                                                           // Frequenz VFO A aus dem Transceiver lesen
delay(dly);                                                           // Verzögerung
       VFOB=(cat_rec_vfo_a/10+RXosz-shift)*10-TXosz-RIT+Splitoffset;  // VFO Freq. TX VFO berechnen und um RIT korrigieren
         unsigned long currentMillis = millis();                      // Wert für aktuellen Timer setzen 
          if (currentMillis - previousMillis >= interval)             // wenn Differenz des aktuellen Timers zu Startwert des Timers > Verzögerungszeit ist, dann:
            {
               previousMillis = currentMillis;                        // Timerstartwert auf Wert des aktuellen Timers setzen
               if (freqmem!=VFOB)                                     // Nur wenn der Zwischenspeicher nicht der TX VFO Frequenz entspricht, dann:
                  {
                   delay(dly);                                        // Verzögerung
                   tx_vfo_b();                                        // VFO mit der neuen TX Frequenz beschreiben
                   freqmem=VFOB;                                      // TX Frequenz in den Zwischenspeicher schreiben
                   delay(dly);                                        // Verzögerung
                  } 
            }
delay(dly);                                                           // Verzögerung
VFOA = cat_rec_vfo_a;                                                 // RX VFO Frequenz vom Transceiver lesen
tx_vfo_a;                                                             // RX Frequenz in den Transceiver schreiben
}


//***********************************************************RIT (RX Korrektur)********************************************************************
void RXkorr()
{
up = digitalRead(uppin);                        // up Taste abfragen
            if (up == LOW)                      // wenn up Taste betätigt
            {
              delay(200);                       // Verzögerung
              RIT=RIT+(sw*10);                  // RIT um Schrittweite*10 erhöhen
              VFOA=((VFOA)+(sw*10));            // RX VFO mit der neuen Frequenz beschreiben
              tx_vfo_a();                       // die Variable (VFOA) in den Transceiver in den VFO A schreiben
              lcd.clear();                      // LCD löschen
              lcd.setCursor(0, 0);              // 1. Zeichen, 1. Zeile
              lcd.print("RIT [Hz]");            // schreiben
              lcd.setCursor(0, 1);              // 1. Zeichen, 2. Zeile
              lcd.print(RIT);                   // schreiben
              delay(500);                       // Verzögerung
              lcd.clear();                      // LCD löschen
            }
down = digitalRead(downpin);                    // down Taste abfragen
            if (down == LOW)                    // wenn down Taste betätigt
            {
              delay(200);                       // Verzögerung
              RIT=RIT-(sw*10);                  // RIT um Schrittweite*10 vermindern
              VFOA=((VFOA)-(sw*10));            // RX VFO mit der neuen Frequenz beschreiben
              tx_vfo_a();                       // die Variable (VFOA) in den Transceiver in den VFO A schreiben
              lcd.clear();                      // LCD löschen
              lcd.setCursor(0, 0);              // 1. Zeichen, 1. Zeile
              lcd.print("RIT [Hz]");            // schreiben
              lcd.setCursor(0, 1);              // 1. Zeichen, 2. Zeile
              lcd.print(RIT);                   // schreiben
              delay(500);                       // Verzögerung
              lcd.clear();                      // LCD löschen
            }
}


// *******************************************************Menüsteuerung****************************************************************************
void menuset()
{
menuState = digitalRead(menuPin);                       // prüfen ob die Menütaste gedrückt ist
  if (menuState != lastmenuState)                       // wenn sich der Menü Status geändert hat
  {
    if (menuState == LOW)                               // wenn Menü Status LOW
    {
      delay(200);                                       // Verzögerung
      menuPushCounter = 5;                              // maximal 4 Menüs verfügbar Menütastenzähler auf 5 setzen
      menubool = true;                                  // menü aktivieren
    }
    if (menuState == HIGH)                              // wenn Menü Status HIGH
    {
      menubool = false;                                 // menü deaktivieren
    }
  }

  if (menubool == true && menuPushCounter == 5)         // wenn Menü aktiviert und Menütastenzähler 5
  {
    menucounter = 0;                                    // Menüzähler auf 0 setzen 
    menu();                                             // Menü starten
  }
lastmenuState = menuState;                              // Menüstatus setzen
}


// *******************************************Menüsteuerung zum Setzen und Speichern der Offsets***************************************************

int menu(){
  lcd.clear();                                    // LCD löschen
  delay(100);                                     // Verzögerung
  while (menucounter < 5)                         // while Menu enspricht der max Anzahl der Menüs
  {
    menuState = digitalRead(menuPin);             // Menü Taste abfragen
    up = digitalRead(uppin);                      // up Taste abfragen
    down = digitalRead(downpin);                  // down Taste abfragen

    if (menuState != lastmenuState)               // wenn sich der Menü Status geändert hat
    {
      lcd.clear();                                // LCD löschen
      if (menuState == LOW)                       // wenn Menü Status LOW
      {
        delay(200);                               // Verzögerung
        menucounter++;                            // Menüzähler um 1 erhöhen
      }
    }
lastmenuState = menuState;                        // letzten Menüstatus auf Menü Status setzen
   
                                                  // zeigt die Menüs

    switch (menucounter) {                        // Menüauswahl
      case 0:                                     // Menü 1 TX Oszillator
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("TX Oszillator [Hz]");          // schreiben
        lcd.setCursor(0, 1);                      // 1. Zeichen, 2. Zeile
        lcd.print(TXosz);                         // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              TXosz=TXosz+(sw*10);                // Wert um Schrittweite*10 erhöhen
              lcd.clear();                        // LCD löschen
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              delay(200);                         // Verzögerung
              TXosz=TXosz-(sw*10);                // Wert um Schrittweite*10 vermindern
              lcd.clear();                        // LCD löschen
            }
        break;                                    // aussteigen, dieses Menü beginnt dann von vorne

      case 1:                                     // Menü 2 RX Oszillator
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("RX Oscillator[Hz/10]");        // schreiben
        lcd.setCursor(0, 1);                      // 1. Zeichen, 2. Zeile
        lcd.print(RXosz);                         // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              RXosz=RXosz+sw;                     // Wert um Schrittweite erhöhen
              lcd.clear();                        // LCD löschen
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              delay(200);                         // Verzögerung
              RXosz=RXosz-sw;                     // Wert um Schrittweite vermindern
              lcd.clear();                        // LCD löschen
            }
        break;                                    // aussteigen, dieses Menü beginnt dann von vorne
        
      case 2:                                     // Menü 2 Split Offset
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("Split Offset [Hz]");           // schreiben
        lcd.setCursor(0, 1);                      // 1. Zeichen, 2. Zeile
        lcd.print(Splitoffset);                   // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              Splitoffset=Splitoffset+(sw*10);    // Wert um Schrittweite erhöhen
              lcd.clear();                        // LCD löschen
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              delay(200);                         // Verzögerung
              Splitoffset=Splitoffset-(sw*10);    // Wert um Schrittweite vermindern
              lcd.clear();                        // LCD löschen
            }
        break;                                    // aussteigen, dieses Menü beginnt dann von vorne

      case 3:                                     // Menü 3 Step Size
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("Step size");                   // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              sw=sw*10;                           // Wert mit 10 multiplizieren
              lcd.clear();                        // LCD löschen
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              delay(200);                         // Verzögerung
              sw=sw/10;                           // Wert durch 10 dividieren
              lcd.clear();                        // LCD löschen
            }
            if (sw*10 > 100000000) {sw=1;}        // max Begrenzung auf 100M
            if (sw < 1) {sw=1;}                   // min Begrenzung auf 1
            if (sw*10 < 1000){                    // Beschriftung und Umrechnung in Hz
              lcd.setCursor(0, 1);                // 1. Zeichen, 2. Zeile
              lcd.print(sw*10);                   // schreiben   
              lcd.setCursor(5, 1);                // 5. Zeichen, 2. Zeile
              lcd.print("Hz");                    // schreiben
            }
            if (sw*10 > 999 && sw*10 < 1000000){  // Beschriftung und Umrechnung in KHz
              lcd.setCursor(0, 1);                // 1. Zeichen, 2. Zeile
              lcd.print(sw*10/1000);              // schreiben
              lcd.setCursor(5, 1);                // 5. Zeichen, 2. Zeile
              lcd.print("KHz");                   // schreiben
            }
            if (sw*10 > 999999 && sw*10 < 1000000000){ // Beschriftung und Umrechnung in MHz
              lcd.setCursor(0, 1);                // 1. Zeichen, 2. Zeile
              lcd.print(sw*10/1000000);           // schreiben
              lcd.setCursor(5, 1);                // 5. Zeichen, 2. Zeile
              lcd.print("MHz");                   // schreiben
            }
        break;                                    // aussteigen, dieses Menü beginnt dann von vorne
        
      case 4:                                     // Menü 4, hier werden die Werte gespeichert und der arduino beginnt loop neu auszuführen
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("store Offsets?");              // schreiben
        lcd.setCursor(0, 1);                      // 1. Zeichen, 2. Zeile
        lcd.print("down or menu= No");            // schreiben
        lcd.setCursor(18, 2);                     // 1. Zeichen, 3. Zeile
        lcd.print("up= Yes");                     // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              EEPROM.put(0, TXosz);               // Wert ins eeprom schreiben (Adresse 0)
              delay(200);                         // Verzögerung
              EEPROM.put(10, RXosz);              // Wert ins eeprom schreiben (Adresse 10)
              delay(200);                         // Verzögerung
              lcd.setCursor(0, 1);                // 1. Zeichen, 1. Zeile
              lcd.clear();                        // LCD löschen
              lcd.print("Offsets stored");        // schreiben
              delay(2000);                        // Verzögerung
              lcd.clear();                        // LCD löschen
              delay(200);                         // Verzögerung
              menucounter = 5;                    // Menü Zähler auf 5 setzen
              menuPushCounter = 0;                // Menütastenzähler auf 0 setzen
              break;                              // aussteigen, zurück zu loop
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              lcd.clear();                        // LCD löschen
              delay(200);                         // Verzögerung
              menucounter = 5;                    // Menü Zähler auf 5 setzen
              menuPushCounter = 0;                // Menütastenzähler auf 0 setzen
              break;                              // aussteigen, zurück zu loop
            }
    }
  }
}


//******************************************************************loop**************************************************************************  
void loop() {         
LCDBeschriftung();                      // LCD Beschriftung ausführen
calculateFrequency();                   // Frequenzberechnung durchführen
displayTXfreq();                        // TX Frequenz ins Display schreiben ausführen
displayRXfreq();                        // RX Frequenz ins Display schreiben ausführen
RXkorr();                               // RIT ausführen
VFOnachziehen();                        // VFO nachziehen ausführen
TXstatus();                             // RX TX Status abfragen ausführen
RXTXBeschriftung();                     // RX TX Statusbeschriftung ausführen
software_reset();                       // Software Reset ausführen
menuset();                              // Menü ausführen
delay(10);                              // Verzögerung
//Serial.println(RXfreq);               // nur für Testzwecke
//Serial.println(TXfreq);               // nur für Testzwecke
//Serial.println(VFOA);                 // nur für Testzwecke
//Serial.println(VFOB);                 // nur für Testzwecke
//Serial.println(cat_rec_vfo_a);        // nur für Testzwecke
//Serial.println(cat_rec_vfo_b);        // nur für Testzwecke
//Serial.println(RIT);                  // nur für Testzwecke
}
